# menu_app.py
import subprocess
import sys
from kivy.app import App
from kivy.uix.screenmanager import ScreenManager, Screen, FadeTransition
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.button import Button
from kivy.uix.gridlayout import GridLayout
from kivy.uix.label import Label
from kivy.uix.textinput import TextInput
from kivy.core.window import Window

# =====================================================
# Load & Save map files
# =====================================================

def load_map():
    try:
        with open("map_layout.txt", "r") as f:
            base_map = [line.strip() for line in f.readlines()]
    except:
        base_map = [
            "00000.....",
            "0...111110",
            "0........0",
            "0...111110",
            "00000.....",
        ]

    old_rows = len(base_map)
    old_cols = len(base_map[0])


    expanded = []

    # expand original rows
    for row in base_map:
        expanded.append(row)

    return expanded

def save_map(map_data):
    with open("map_layout.txt", "w") as f:
        for row in map_data:
            f.write(row + "\n")


# =====================================================
# Menu Screen
# =====================================================

class MenuScreen(Screen):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        layout = BoxLayout(orientation='vertical', spacing=20, padding=40)

        start_btn = Button(text='Start Game', size_hint=(1, 0.25))
        editor_btn = Button(text='Map Editor', size_hint=(1, 0.25))
        exit_btn = Button(text='Exit', size_hint=(1, 0.25))

        start_btn.bind(on_release=self.start_game)
        editor_btn.bind(on_release=self.open_editor)
        exit_btn.bind(on_release=self.exit_app)

        layout.add_widget(start_btn)
        layout.add_widget(editor_btn)
        layout.add_widget(exit_btn)
        self.add_widget(layout)

    def start_game(self, instance):
        # Runs your pygame script as external process
        subprocess.Popen([sys.executable, "main.py"])

    def open_editor(self, instance):
        self.manager.current = 'editor'

    def exit_app(self, instance):
        App.get_running_app().stop()


# =====================================================
# Map Editor Screen
# =====================================================

class MapEditorScreen(Screen):
    TILE_TYPES = ["0", "1", "2", "."]  # cycle through these

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self.map_data = load_map()
        self.rows = len(self.map_data)
        self.cols = len(self.map_data[0])

        root = BoxLayout(orientation="vertical", padding=100, spacing=100)

        # --- grid editor ---
        self.grid = GridLayout(cols=self.cols, rows=self.rows, spacing=2)
        self.tiles = []

        for y in range(self.rows):
            row_buttons = []
            for x in range(self.cols):
                tile_char = self.map_data[y][x]
                btn = Button(text=tile_char)
                btn.bind(on_release=self.change_tile)
                self.grid.add_widget(btn)
                row_buttons.append(btn)
            self.tiles.append(row_buttons)

        root.add_widget(self.grid)

        # --- bottom controls ---
        controls = BoxLayout(size_hint=(1, 0.2), spacing=20)

        save_btn = Button(text="Save")
        back_btn = Button(text="Back to Menu")

        save_btn.bind(on_release=self.save)
        back_btn.bind(on_release=self.back)

        controls.add_widget(save_btn)
        controls.add_widget(back_btn)
        root.add_widget(controls)

        self.add_widget(root)

    def change_tile(self, btn):
        """Cycle through characters like: 0 → 1 → 2 → . → 0"""
        current = btn.text
        i = self.TILE_TYPES.index(current)
        next_type = self.TILE_TYPES[(i + 1) % len(self.TILE_TYPES)]
        btn.text = next_type

    def save(self, instance):
        # Convert buttons back to text map
        new_map = []
        for row in self.tiles:
            new_map.append("".join([t.text for t in row]))
        save_map(new_map)
        print("Map saved to map_layout.txt")

    def back(self, instance):
        self.manager.current = "menu"


# =====================================================
# The App
# =====================================================

class MenuApp(App):
    subprocess.Popen([sys.executable, "main.py"])
    def build(self):
        Window.clearcolor = (0, 0, 0, 1)
        sm = ScreenManager(transition=FadeTransition())
        sm.add_widget(MenuScreen(name='menu'))
        sm.add_widget(MapEditorScreen(name='editor'))
        return sm


if __name__ == '__main__':
    MenuApp().run()
